// Author: Kevin Scroggins
// Email: nitro404@hotmail.com
// Date: September 18, 2008

#ifndef _VECTOR_CPP
#define _VECTOR_CPP

#include "Vector.h"

using namespace std;

// constructors
template <class T>
Vector<T>::Vector(int initial_size) {
	// initialize local variables
	this->capacity = (initial_size < 1) ? 10 : initial_size; // ensure the initial size is valid
	this->numberOfElements = 0;
	this->elements = new T*[this->capacity];
}

// copy constructor
template <class T>
Vector<T>::Vector(const Vector<T> &v) {
	if(this != &v) {
		// copy local variable data
		this->capacity = v.capacity;
		this->numberOfElements = v.numberOfElements;
		this->elements = new T*[this->capacity];
		
		// copy the elements
		for(int i=0;i<this->numberOfElements;i++) {
			this->elements[i] = v.elements[i];
		}
	}
}

// assignment constructor
template <class T>
Vector<T> &Vector<T>::operator = (const Vector<T> &v) {
	if(this != &v) {
		//delete old element storage container
		delete [] this->elements;
		
		// copy local variable data
		this->capacity = v.capacity;
		this->numberOfElements = v.numberOfElements;
		this->elements = new T*[this->capacity];
		
		// copy the elements
		for(int i=0;i<this->numberOfElements;i++) {
			this->elements[i] = v.elements[i];
		}
	}
	return *this;
}

// destructor
template <class T>
Vector<T>::~Vector(void) {
	delete [] this->elements;
}

// add an element to the vector
template <class T>
Vector<T> &Vector<T>::add(T & x) {
	// add the element after the last element in the vector
	add(x, numberOfElements);
	
	return *this;
}

// insert an element to the vector at a specific index position
template <class T>
Vector<T> &Vector<T>::add(T & x, int index) {
	// if the vector is full, create a new vector with the same elements with twice the capacity
	if(numberOfElements == capacity) {
		this->capacity *= 2;
		T **temp = new T*[this->capacity];
		
		for(int i=0;i<this->numberOfElements;i++)
			temp[i] = this->elements[i];
		
		delete [] this->elements;
		
		this->elements = temp;
	}
	
	for(int j=this->numberOfElements - 1;j>index;j--) { // loop through elements after the specified index
		this->elements[j] = this->elements[j-1]; // shift each element to the right one position
	}
	
	// add the element to the vector at the specified index
	this->numberOfElements++;
	this->elements[index] = &x;
	
	return *this;
}

// remove an element from the vector
template <class T>
Vector<T> &Vector<T>::remove(const T & x) {
	// get the index of the element
	int index = indexOf(x);

	if(index < 0) return *this;
	
	// if there is at least one element and the vector is half-full,
	// create a new vector with the same elements that is half the size
	if(numberOfElements > 1 && numberOfElements == (capacity / 2) ) {
		capacity /= 2;
		T **temp = new T*[this->capacity];
		
		for(int i=0;i<this->numberOfElements;i++) {
			temp[i] = this->elements[i];
		}
		
		delete [] this->elements;
		
		this->elements = temp;
	}
	
	// shift the elements after the position of the removed element to the left one position
	for(int i=index;i<this->numberOfElements - 1;i++) {
		this->elements[i] = this->elements[i+1];
	}
	
	this->numberOfElements--;
	
	return *this;
}

// return the element at a specific index
template <class T>
T &Vector<T>::elementAt(int index) const {
	return *(this->elements[index]);
}

// return the number of elements in the vector
template <class T>
int Vector<T>::size() const {
	return this->numberOfElements;
}

// return whether or not there are any elements in the vector
template <class T>
bool Vector<T>::isEmpty() const {
	return this->numberOfElements == 0;
}

// clear all elements out of the vector
template <class T>
void Vector<T>::clear() {
	this->numberOfElements = 0;
}

// return if the vector contains an element or not
template <class T>
bool Vector<T>::contains(const T & x) const {
	// loop through all elements and return true if there is a matching element
	for(int i=0;i<this->numberOfElements;i++) {
		if(*(this->elements[i]) == x) {
			return true;
		}
	}

	return false;
}

// return the first index of a specific element in the vector
template <class T>
int Vector<T>::indexOf(const T & x) const {
	return indexOf(x, 0);
}

// return the next index of a specific element in the vector after a specified index
template <class T>
int Vector<T>::indexOf(const T & x, int index) const {
	int elementIndex = -1;

	// if the search index is out of bounds, change it to 0
	if(index < 0 || index >= this->numberOfElements) {
		index = 0;
	}

	// loop through the elements and load the index, i into the element index if a match is found
	for(int i=0;i<this->numberOfElements;i++) {
		if(*(this->elements[i]) == x) {
			elementIndex = i;
		}
	}

	return elementIndex; // return the element's index (-1 if it is not found)
}

// equality operator override
template <class T>
bool Vector<T>::operator == (const Vector<T> &v) const {
	// compare all elements
	for(int i=0;i<this->numberOfElements;i++) {
		if(*(this->elements[i]) != *v.elements[i]) {
			return false;
		}
	}

	return true;
}

// inequality operator override
template <class T>
bool Vector<T>::operator != (const Vector<T> &v) const {
	return !operator == (v);
}

// print to stream method
template <class T>
void Vector<T>::printOn(ostream &o) const {
	// loop through all elements and print them to the stream in the form [a, b, c]
	o << "[";
	for(int i=0;i<this->numberOfElements;i++) {
		o << *(this->elements[i]);
		if(i<this->numberOfElements - 1)
			o << ", ";
	}
	o << "]";
}

// print to stream operator override
template <class T>
ostream &operator << (ostream &o, const Vector<T> &v) {
	v.printOn(o);
	return o;
}

#endif